import logging
import json
from telegram import (
    InlineKeyboardButton,
    InlineKeyboardMarkup,
    Update,
    InputMediaPhoto
)
from telegram.ext import (
    ApplicationBuilder,
    CommandHandler,
    CallbackQueryHandler,
    ContextTypes
)

# Enable logging for debugging purposes.
logging.basicConfig(
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    level=logging.INFO
)

def load_questions():
    """
    Load questions, answers, and image paths from the external JSON file.
    """
    with open("questions.json", "r") as file:
        data = json.load(file)
    return data.get("Questions", [])

def build_start_keyboard():
    """
    Build an inline keyboard for the start menu based on the JSON file.
    """
    questions = load_questions()
    keyboard = []
    for idx, q in enumerate(questions):
        # Assume keys starting with "Question" hold the question text.
        question_text = next((value for key, value in q.items() if key.startswith("Question")), "Question")
        keyboard.append([InlineKeyboardButton(question_text, callback_data=f"question_{idx}")])
    return InlineKeyboardMarkup(keyboard)

async def send_start_menu(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """
    Sends the start menu as a photo message using a default image.
    The caption shows a prompt and the inline keyboard lists available questions.
    """
    keyboard = build_start_keyboard()
    default_caption = "Please choose a question:"
    chat_id = update.effective_chat.id
    # Send a photo message so that we can later edit it with edit_message_media.
    message = await context.bot.send_photo(
        chat_id=chat_id,
        photo=open("qanda.jpg", "rb"),
        caption=default_caption,
        reply_markup=keyboard
    )
    return message

async def start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """
    Handler for the /start command.
    Sends the starting menu.
    """
    await send_start_menu(update, context)

async def button_handler(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """
    Handles all button callbacks.
    
    • When a question button is pressed, the bot retrieves the corresponding question,
      then edits the same message (using edit_message_media) with the answer image and caption.
    
    • When the "Ask another question" button is pressed, the bot clears in‑RAM user and chat data,
      and then edits that same message back to the starting menu.
    """
    query = update.callback_query
    await query.answer()  # Acknowledge the callback

    data = query.data

    if data.startswith("question_"):
        # Extract the question index from the callback data.
        index = int(data.split("_")[1])
        questions = load_questions()
        
        if index < len(questions):
            question_data = questions[index]
            # Assume keys starting with "Answer" hold the answer text.
            answer_text = next((value for key, value in question_data.items() if key.startswith("Answer")), "No answer available")
            image_path = question_data.get("image", None)
            # Build a keyboard with the "Ask another question" button.
            keyboard = [[InlineKeyboardButton("Ask another question", callback_data="restart")]]
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            if image_path:
                # Open the answer image and create a new media object.
                with open(image_path, "rb") as photo:
                    new_media = InputMediaPhoto(media=photo, caption=answer_text)
                    # Edit the existing message to display the new photo and caption.
                    await query.edit_message_media(media=new_media, reply_markup=reply_markup)
            else:
                # If no image is provided, fall back to editing the caption only.
                await query.edit_message_caption(caption=answer_text, reply_markup=reply_markup)

    elif data == "restart":
        # Clear temporary in‑RAM data (this does not delete any chat history).
        context.user_data.clear()
        context.chat_data.clear()
        # Rebuild the start menu inline keyboard and default caption.
        keyboard = build_start_keyboard()
        default_caption = "Please choose a question:"
        with open("qanda.jpg", "rb") as photo:
            new_media = InputMediaPhoto(media=photo, caption=default_caption)
            # Edit the same message to revert to the start menu.
            await query.edit_message_media(media=new_media, reply_markup=keyboard)

def main():
    """
    Create and run the Telegram bot application.
    Replace 'YOUR_BOT_TOKEN_HERE' with your actual bot token.
    """
    application = ApplicationBuilder().token("YOUR_TOKEN_HERE").build()

    application.add_handler(CommandHandler("start", start))
    application.add_handler(CallbackQueryHandler(button_handler))

    application.run_polling()

if __name__ == "__main__":
    main()
